// Content script for CB Vulgarities extension
console.log('CB Vulgarities Extension: Content script loaded on', window.location.href);

// Initialize extension
(function() {
  'use strict';

  console.log('CB Vulgarities Extension: Initializing content script');

  // Send message to background script to confirm loading
  chrome.runtime.sendMessage(
    { action: 'contentScriptLoaded', url: window.location.href },
    (response) => {
      console.log('CB Vulgarities Extension: Background script response', response);
    }
  );

  // State management
  let currentMode = 'pre-show';
  let isRunning = false;

  // Load state from storage
  chrome.storage.local.get(['currentMode', 'isRunning'], (result) => {
    if (result.currentMode) currentMode = result.currentMode;
    if (result.isRunning !== undefined) isRunning = result.isRunning;
    console.log('CB Vulgarities Extension: Loaded state', { currentMode, isRunning });
    updateButtonState();
  });

  // Function to save state
  function saveState() {
    chrome.storage.local.set({ currentMode, isRunning }, () => {
      console.log('CB Vulgarities Extension: State saved', { currentMode, isRunning });
      // Notify background script of state change
      chrome.runtime.sendMessage({
        action: 'stateChanged',
        state: { currentMode, isRunning }
      });
    });
  }

  // Wait for the tab-row div to be available
  function injectControls() {
    // Check if already injected
    if (document.getElementById('cb-vulgarities-controls')) {
      console.log('CB Vulgarities Extension: Controls already injected');
      return;
    }

    // Find all tab-row divs (there might be multiple - theatre mode and split mode)
    const tabRows = document.querySelectorAll('[id="tab-row"]');

    if (tabRows.length === 0) {
      console.log('CB Vulgarities Extension: tab-row not found, retrying...');
      setTimeout(injectControls, 500);
      return;
    }

    // Find the split mode tab-row by checking for the split-mode chat container
    let tabRow = null;

    for (const row of tabRows) {
      // Check if this tab-row is inside a split-mode context
      // Look for the msg-list-wrapper-split container which indicates split mode
      const splitModeContainer = row.closest('.msg-list-wrapper-split') ||
                                 document.querySelector('.msg-list-wrapper-split');

      if (splitModeContainer) {
        // This is the split mode tab-row
        tabRow = row;
        break;
      }
    }

    if (!tabRow) {
      console.log('CB Vulgarities Extension: split mode tab-row not found, retrying...');
      setTimeout(injectControls, 500);
      return;
    }

    console.log('CB Vulgarities Extension: Found split mode tab-row, injecting controls');

    // Create container for our controls
    const controlsContainer = document.createElement('div');
    controlsContainer.id = 'cb-vulgarities-controls';
    controlsContainer.style.cssText = `
      position: absolute;
      right: 8px;
      top: 50%;
      transform: translateY(-50%);
      display: flex;
      align-items: center;
      gap: 8px;
      z-index: 1000;
    `;

    // Create dropdown
    const dropdown = document.createElement('select');
    dropdown.id = 'cb-vulgarities-mode-select';
    dropdown.style.cssText = `
      padding: 2px 6px;
      font-size: 12px;
      border-radius: 4px;
      border: 1px solid rgba(255, 255, 255, 0.3);
      background: rgba(255, 255, 255, 0.1);
      color: inherit;
      cursor: pointer;
      outline: none;
      font-family: inherit;
    `;

    // Add dropdown options
    const modes = ['pre-show', 'blowjob', 'cumshow'];
    modes.forEach(mode => {
      const option = document.createElement('option');
      option.value = mode;
      option.textContent = mode;
      option.selected = mode === currentMode;
      dropdown.appendChild(option);
    });

    // Create play/stop button
    const button = document.createElement('button');
    button.id = 'cb-vulgarities-toggle-button';
    button.style.cssText = `
      padding: 2px 8px;
      font-size: 12px;
      border-radius: 4px;
      border: 1px solid rgba(255, 255, 255, 0.3);
      background: rgba(255, 255, 255, 0.1);
      color: inherit;
      cursor: pointer;
      outline: none;
      font-family: inherit;
      min-width: 45px;
      transition: all 0.2s ease;
    `;

    // Function to update button state
    function updateButtonState() {
      button.textContent = isRunning ? 'Stop' : 'Play';
      button.style.background = isRunning ? 'rgba(255, 100, 100, 0.3)' : 'rgba(100, 255, 150, 0.3)';
    }

    updateButtonState();

    // Event listeners
    dropdown.addEventListener('change', (e) => {
      currentMode = e.target.value;
      console.log('CB Vulgarities Extension: Mode changed to', currentMode);
      saveState();
    });

    button.addEventListener('click', () => {
      isRunning = !isRunning;
      console.log('CB Vulgarities Extension: Running state changed to', isRunning);
      updateButtonState();
      saveState();
    });

    // Add hover effects
    dropdown.addEventListener('mouseenter', () => {
      dropdown.style.background = 'rgba(255, 255, 255, 0.15)';
    });
    dropdown.addEventListener('mouseleave', () => {
      dropdown.style.background = 'rgba(255, 255, 255, 0.1)';
    });

    button.addEventListener('mouseenter', () => {
      button.style.transform = 'scale(1.05)';
    });
    button.addEventListener('mouseleave', () => {
      button.style.transform = 'scale(1)';
    });

    // Append controls to container
    controlsContainer.appendChild(dropdown);
    controlsContainer.appendChild(button);

    // Inject into tab-row
    tabRow.style.position = 'relative';
    tabRow.appendChild(controlsContainer);

    console.log('CB Vulgarities Extension: Controls injected successfully');
  }

  // Make updateButtonState available globally within this scope
  function updateButtonState() {
    const button = document.getElementById('cb-vulgarities-toggle-button');
    if (button) {
      button.textContent = isRunning ? 'Stop' : 'Play';
      button.style.background = isRunning ? 'rgba(255, 100, 100, 0.3)' : 'rgba(100, 255, 150, 0.3)';
    }
  }

  // Start injection when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', injectControls);
  } else {
    injectControls();
  }

  console.log('CB Vulgarities Extension: Content script initialization complete');
})();
